/*	Copyright 2002 by Eric Postpischil, http://edp.org.
	See license information in index.html.
*/


/*	Coordinate system:

	We use an unusual coordinate system.  We tile the plane with
	equilateral triangles.  These triangles alternate pointing
	up and down.  Because this gives them a rectangular pattern,
	we can identify them with row and column numbers.  This is
	convenient for representation in a program.

	Translations that map up-pointing triangles to other up-pointing
	triangles (and similarly for down-pointing triangles) are
	easily represented by shifting rows or columns.  Reflections
	can be represented by negating a coordinate.

	Rotations are more difficult.  We have two classes of
	rotations to consider.  First, there are 180-degree rotations
	that map upward triangles into downward triangles.  Second,
	there are 120-degree rotations that map upward triangles into
	upward triangles.

	The former, 180-degree rotations, can be represented by negating
	both coordinates and translating by the width of one column.
	(See the rotate180 source code.)

	The latter, 120-degree rotations, can be performed by mapping
	row and column numbers to Cartesian coordinates for the center
	of the corresponding triangle, performing a rigid geometric
	rotation on the center, and mapping the result back to row and
	column numbers.  (See the rotate120 source code for elaboration.)
*/


// Define the geometric width and height of a triangle.
#define	WIDTH	.5f
#define	HEIGHT	.8660254037844386f


// A point is defined by its coordinates.
typedef struct {
		short	x, y;
}	Point;

/*	We use small integers to label the pieces, so they can be
	distinguished when combined into a goal shape.  Particular values
	also indicate empty spaces and goal spaces.

	We also use these integers to select colors for the pieces.  Our
	color array is set to 256 elements so it may be indexed by an
	unsigned char without further effort.  If the number of colors is
	changed, expressions that prepare an index for the array must be
	changed.
*/
typedef unsigned char	PieceID;

//Define values to be used to mark types of spaces.
enum {
	EMPTY,				// an empty space, not part of goal or piece
						// EMPTY must be zero so calloc initializes as desired
	GOAL,				// a goal space to be filled in
	PIECE,				// first number for an ordinary space
	NUM_COLORS = 256	// number of colors available, which limits number of pieces
};


// A shape is a bunch of points.
typedef struct {
		short			width, height;	// extent of shape
		unsigned short	numPoints;		// number of points in shape
		char			parity;			// orientation of triangle at shape origin
										// 0 is downward-pointing.  1 is upward.
		Bool			isVolatile;		// shape changes asynchronously
		// (a volatile shape must be completely redrawn if any part is redrawn)
		PieceID			*map;			// array of piece identifiers
		/*	map must be unsigned char to index color array.
			(Just in case we ever exceed 254 pieces!)
		*/
}	Shape;								// See Shapes.cpp for more information.

// Define a list of shapes.
typedef struct shapelist {
		struct shapelist	*next;
		Shape				*shape;
}	ShapeList;

// Define location of point (i,j) within shape map.
#define	SPOT(shape, i, j)	((shape)->map[(shape)->width*(j)+(i)])

// A piece is a list of points.
typedef struct {
		short			width, height;	// extent of piece
		Point			*points;		// list of points (triangles) in piece
		unsigned short	numPoints;		// number of points in list
		char			parity;			// orientation of triangle at piece origin
										// 0 is downward-pointing.  1 is upward.
		PieceID			ID;				// identifier of piece
}	Piece;								// See Shapes.cpp for more information.

// Define a list of pieces.
typedef struct piecelist {
		struct piecelist	*next;
		Piece				*piece;
}	PieceList;

// Define a set of pieces as a list of the transforms of pieces.
typedef struct pieces {
		struct pieces	*next;	// next list in list
		PieceList		*list;	// list of transformations of one piece
		Bool			used;	// flag to indicate piece is used
}	Pieces;